<?php
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/message_logging.php';

function sendSMS($phoneNumber, $message) {
    global $pdo;
    try {
        // Get SMS settings from database
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Check if SMS is enabled
        if (($settings['enable_sms'] ?? '0') !== '1') {
            logMessageAttempt('sms', $phoneNumber, $message, 'failed', 'SMS notifications are not enabled');
            throw new Exception('SMS notifications are not enabled');
        }

        // Get provider and credentials
        $provider = $settings['sms_provider'] ?? '';
        $apiKey = $settings['sms_api_key'] ?? '';
        $apiSecret = $settings['sms_api_secret'] ?? '';
        $senderId = $settings['sms_sender_id'] ?? '';

        // Validate required settings
        if (empty($provider) || empty($apiKey) || empty($apiSecret) || empty($senderId)) {
            logMessageAttempt('sms', $phoneNumber, $message, 'failed', 'SMS settings are not properly configured');
            throw new Exception('SMS settings are not properly configured');
        }

        // Format phone number (remove spaces, ensure it starts with +)
        $phoneNumber = preg_replace('/\s+/', '', $phoneNumber);
        if (!str_starts_with($phoneNumber, '+')) {
            $phoneNumber = '+' . $phoneNumber;
        }

        // Send SMS based on provider
        switch ($provider) {
            case 'africas_talking':
                $result = sendViaAfricasTalking($phoneNumber, $message, $apiKey, $apiSecret, $senderId);
                logMessageAttempt('sms', $phoneNumber, $message, 'success', '', 'Sent via Africa\'s Talking');
                return $result;
            case 'twilio':
                $result = sendViaTwilio($phoneNumber, $message, $apiKey, $apiSecret, $senderId);
                logMessageAttempt('sms', $phoneNumber, $message, 'success', '', 'Sent via Twilio');
                return $result;
            default:
                logMessageAttempt('sms', $phoneNumber, $message, 'failed', 'Invalid SMS provider');
                throw new Exception('Invalid SMS provider');
        }
    } catch (Exception $e) {
        logMessageAttempt('sms', $phoneNumber, $message, 'failed', $e->getMessage());
        error_log("SMS Error: " . $e->getMessage());
        throw $e;
    }
}

function sendViaAfricasTalking($phoneNumber, $message, $apiKey, $apiSecret, $senderId) {
    $url = 'https://api.africastalking.com/version1/messaging';
    $data = [
        'username' => $apiKey,
        'to' => $phoneNumber,
        'message' => $message,
        'from' => $senderId
    ];

    $headers = [
        'Accept: application/json',
        'Content-Type: application/x-www-form-urlencoded',
        'apiKey: ' . $apiSecret
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode !== 201) {
        throw new Exception('Failed to send SMS via Africa\'s Talking: ' . $response);
    }

    return true;
}

function sendViaTwilio($phoneNumber, $message, $accountSid, $authToken, $fromNumber) {
    $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json";
    $data = [
        'To' => $phoneNumber,
        'From' => $fromNumber,
        'Body' => $message
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERPWD, $accountSid . ":" . $authToken);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode !== 201) {
        throw new Exception('Failed to send SMS via Twilio: ' . $response);
    }

    return true;
}

// Function to test SMS configuration
function testSMSSettings() {
    global $pdo;
    try {
        // Get SMS settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        if (($settings['enable_sms'] ?? '0') !== '1') {
            logMessageAttempt('sms', 'test', 'Test SMS', 'failed', 'SMS notifications are not enabled');
            return ['success' => false, 'message' => 'SMS notifications are not enabled'];
        }

        $provider = $settings['sms_provider'] ?? '';
        $apiKey = $settings['sms_api_key'] ?? '';
        $apiSecret = $settings['sms_api_secret'] ?? '';
        $senderId = $settings['sms_sender_id'] ?? '';

        if (empty($provider) || empty($apiKey) || empty($apiSecret) || empty($senderId)) {
            logMessageAttempt('sms', 'test', 'Test SMS', 'failed', 'SMS settings are not properly configured');
            return ['success' => false, 'message' => 'SMS settings are not properly configured'];
        }

        // Send a test message
        $testMessage = "This is a test message from your POS system. If you receive this, your SMS configuration is working correctly.";
        $testNumber = $settings['sms_test_number'] ?? '';

        if (empty($testNumber)) {
            logMessageAttempt('sms', 'test', 'Test SMS', 'failed', 'Test phone number not configured');
            return ['success' => false, 'message' => 'Test phone number not configured'];
        }

        sendSMS($testNumber, $testMessage);
        logMessageAttempt('sms', $testNumber, $testMessage, 'success', '', 'Test message sent successfully');
        return ['success' => true, 'message' => 'Test message sent successfully'];

    } catch (Exception $e) {
        logMessageAttempt('sms', 'test', 'Test SMS', 'failed', $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
} 